"use strict";
var fm = {};
fm.unitmap = {
	// Flow
	"l/s": 1000,
	"l/min": 60000,
	"l/h": 3600000,
	"m³/s": 1,
	"m³/h": 3600,
	"m³/m": 2629800,
	"m³/y": 31557600,
	// Power
	"W": 1,
	"kW": 1 / 1000,
	"MW": 1 / 1000000,
	"kWh/m": 0.7305,
	"kWh/y": 8766 / 1000,
	"MWh/y": 8766 / 1000000,
	"hk": 1.3596 / 1000,
	// Energy
	"J": 1,
	"kJ": 1 / 1000,
	"MJ": 1 / 1000000,
	"Wh": 1 / 3600,
	"kWh": 1 / 3600000,
	"MWh": 1 / 3600000000,
	// Pressure
	"Pa": 1,
	"hPa": 1 / 100,
	"kPa": 1 / 1000,
	"bar": 1 / 100000,
	"mVp": 1 / 10197.2,
	"mmHg": 133.32,
	"cmHg": 13.332,
	// Torque
	"Nm": 1,
	"Inlb": 1 / 0.11298,
	"Ftlb": 1 / 1.3558,
	// Partials
	"%": 100,
	"‰": 1000,
	"ppm": 1000000,
	// Time
	"sek": 1,
	"min": 1 / 60,
	"h": 1 / 3600,
	"dygn": 1 / 86400,
	"år": 1 / 31557600,
	// Length
	"m": 1,
	"dm": 10,
	"cm": 100,
	"mm": 1000,
	// Misc
	"SFP": 1 / 1000,
	"g/kg": 1000,
	"%rH": 100,
	"m³/MWh": 1 / 9000000,
	"ΔT": 7740000000,
	"Kv": 36000
}

// Specific heat capacity in J/m3K
fm.heatcap = {
	"water20": 4174472,
	"water60": 4114692,
	"prop60": 3540539,
	"prop40": 3833690,
	"prop30": 3959764,
	"etanol": 4162830,
	"air": 1218
}

// Viscosity for fluids depending on temperature in C
fm.viscosity = {
	water: (t) => { return 0.00000002939 * Math.exp(507.88 / (t + 123.84)); },
	air: (t) => { return 0.0000000000002791 * (t + 273.14) ** 0.7355; },
	prop40: (t) => { return 0.000000042; } // Not correct, value for pure
}

// Density for fluids depending on temperature in C
fm.density = {
	water: (t) => { return 999.85308 + (0.063269 * t) - (0.008523829 * t ** 2) + (0.00006943248 * t ** 3) - (0.000000382121 * t ** 4); },
	air: (t) => { return 2936.3985 / (8.3145 * (t + 273.15)); }, // Value for dry air at atmospheric pressure
	prop40: (t) => { return 1036; } // Missing formula
}

fm.kv = {
	"STAD 10": {x: [1, 1.5, 2, 2.5, 3, 3.5, 4], y: [0.091, 0.134, 0.264, 0.461, 0.799, 1.22, 1.36]},
	"STAD 15": {x: [0.5, 1, 1.5, 2, 2.5, 3, 3.5, 4], y: [0.136, 0.226, 0.347, 0.618, 0.931, 1.46, 2.07, 2.56]},
	"STAD 20": {x: [0.5, 1, 1.5, 2, 2.5, 3, 3.5, 4], y: [0.533, 0.781, 1.22, 1.95, 2.71, 3.71, 4.51, 5.39]},
	"STAD 25": {x: [0.5, 1, 1.5, 2, 2.5, 3, 3.5, 4], y: [0.599, 1.03, 2.13, 3.64, 5.26, 6.65, 7.79, 8.59]},
	"STAD 32": {x: [0.5, 1, 1.5, 2, 2.5, 3, 3.5, 4], y: [1.19, 2.09, 3.36, 5.22, 7.77, 9.82, 11.9, 14.2]},
	"STAD 40": {x: [0.5, 1, 1.5, 2, 2.5, 3, 3.5, 4], y: [1.89, 3.40, 4.74, 6.25, 9.16, 12.8, 16.2, 19.3]},
	"STAD 50": {x: [0.5, 1, 1.5, 2, 2.5, 3, 3.5, 4], y: [2.62, 4.1, 6.76, 11.4, 15.8, 21.5, 27, 32.3]},
	"STV 10": {x: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10], y: [0.11, 0.18, 0.27, 0.41, 0.65, 1.02, 1.78, 2.3, 2.6, 2.8]},
	"STV 15": {x: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10], y: [0.18, 0.32, 0.45, 0.62, 0.86, 1.17, 1.62, 2.55, 3.15, 3.55]},
	"STV 20": {x: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10], y: [0.34, 0.6, 0.83, 1.13, 1.55, 2.1, 2.9, 3.85, 4.5, 5.1]},
	"STV 25": {x: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10], y: [0.48, 0.77, 1.03, 1.5, 2.3, 3.6, 5, 6.5, 7.9, 8.8]},
	"STV 32": {x: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10], y: [0.79, 1.32, 1.8, 2.7, 4.1, 5.9, 7.8, 9.7, 11.5, 13.10]},
	"STV 40": {x: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10], y: [1.2, 2.05, 2.8, 4.1, 6.2, 8.9, 12, 14.7, 17.1, 19.5]},
	"STV 50": {x: [1, 2, 3, 4, 5, 6, 7, 8, 9, 10], y: [2, 3.6, 5.2, 7.6, 11.9, 16.7, 21.2, 25, 28.6, 31.5]},
	"STAF 50": {x: [0.5, 1, 1.5, 2, 2.5, 3, 3.5, 4], y: [2.56, 4.2, 7.2, 11.7, 16.2, 21.5, 26.5, 33]},
	"STAF 65": {x: [0.5, 1, 1.5, 2, 2.5, 3, 3.5, 4, 4.5, 5, 5.5, 6, 6.5, 7, 7.5, 8], y: [1.02, 2.39, 3.77, 5.18, 6.52, 8.18, 11.6, 18.6, 29.9, 39.6, 47.9, 57.5, 66.3, 74.2, 80, 85]},
	"STAF 80": {x: [0.5, 1, 1.5, 2, 2.5, 3, 3.5, 4, 4.5, 5, 5.5, 6, 6.5, 7, 7.5, 8], y: [2.33, 4.25, 6.2, 8.47, 11.4, 15, 20.8, 29.9, 43.3, 57.5, 69.6, 81.2, 92.8, 104, 114, 123]}
}

function openPage(target) {
	document.activeElement.blur();
	for(const e of document.getElementsByClassName("page")) {
		e.style.display = "none";
	}
	document.getElementById(target).style.display = "block";

	if (document.documentElement.clientWidth < 881) {
		document.getElementById("sidebar").style.display = "none";
	}

	for (const e of document.getElementsByClassName("page-link")) {
		if (e.dataset.target == target) {
			e.classList.add("page-link-active");
			document.getElementById("active-page").innerText = e.innerText;
		} else {
			e.classList.remove("page-link-active");
		}
	}
	window.scrollTo(0, 0);
	if (fm.wasLastHome) {
		history.pushState({},"");
	}
	fm.wasLastHome = (target == "startMenu");
}

function openSidebar() {
	document.getElementById("sidebar").style.display = "block";
}

function closeSidebar() {
	document.getElementById("sidebar").style.display = "none";
}

function dataListener(elem) {
	for (let e of elem.querySelectorAll(".dataUnitSelect")) {
		e.parentNode.dataset.unit = e.options[e.selectedIndex].text;
		e.onchange = () => {
			e.parentNode.dataset.unit = e.options[e.selectedIndex].text;
			if (e.parentNode.querySelector(".dataInput").value) {
				e.parentNode.querySelector("button").click();
			}
		}
	}
}

function loadStatic() {
	fetch("dut.html")
		.then(response => response.text())
		.then(text => document.getElementById("dut").innerHTML = text);
	fetch("guidelines.html")
		.then(response => response.text())
		.then(text => document.getElementById("guidelines").innerHTML = text);
	fetch("yepresets.html")
		.then(response => response.text())
		.then(text => document.getElementById("yepresets").innerHTML = text)
		.then(function() { yEPListeners() });
}

function fPSettings() {
	let valves = document.getElementById("valveFlowPresets");
	for (const [key, def] of Object.entries(fm.kv)) {
		let opt = document.createElement("option");
		opt.value = key;
		opt.innerHTML = key;
		valves.appendChild(opt);
	}
}

function fPSlider() {
	const set = fm.kv[document.getElementById("valveFlowPresets").value];
	const slider = document.getElementById("valveFlowSetting");
	slider.min = set.x[0];
	slider.max = set.x[set.x.length - 1];
	slider.value = set.x[0];
	putn("#valveFlowSet", set.x[0]);
	fPSet();
}

function fPSet() {
	const set = fm.kv[document.getElementById("valveFlowPresets").value];
	putn("#valveFlowKv", new Converter(curve(set, getn("#valveFlowSet")), "Kv").value);
}

// Finds output on a multi point curve from the input
// def is an object where x and y are two arrays with the points
function curve(def, input) {
	if (input < def.x[0]) {return def.y[0]}
	for (let i = 0; i < (def.x.length - 1); i++) {
		if (input >= def.x[i] && input < def.x[i + 1]) {
			return ((def.y[i + 1] - def.y[i]) / (def.x[i + 1] - def.x[i])) * (input - def.x[i]) + def.y[i];
		}
	}
	return def.y[def.x.length - 1]
}

function yEPListeners() {
	document.getElementById("yearEnergyPresets").addEventListener("change", function() {
		document.getElementById("yearEnergyYearAverage").value = this.value;
	});
}

function negate(event) {
	let input = event.target.parentNode.querySelector(".dataInput");
	input.value = Number(input.value) * -1;
}

function nextInput(event) {
	if (event.code === "Enter") {
		const currentInput = event.target;
		if (currentInput.nodeName != "INPUT") return;

		const form = currentInput.form;
		if (!form) return;

		const formInputs = Array.from(form).filter(
			(element) => element.nodeName === "INPUT"
		)

		let nextInputIndex = formInputs.indexOf(currentInput) + 1;
		if (nextInputIndex >= formInputs.length) {
			nextInputIndex = 0;
		}

		formInputs[nextInputIndex].focus();
		event.preventDefault();
	}
}

// Conversions
class Converter {
	constructor(value, unit) {
		this.value = (unit && fm.unitmap[unit]) ? value / fm.unitmap[unit] : value;
	}

	to_unit(unit) {
		return this.value * (fm.unitmap[unit] || 1);
	}
}

// Fetches a value from the selector, if the selector returns
// multiple element, returns the average of all elements.
// If the element has a data-unit type, the number is converted to the 
// base unit. For elements with numeric value it returns 0.
function getn(selector, avg = true) {
	let sum = 0;
	let num = 0;
	for(const e of document.querySelectorAll(selector)) {
		let n = Number(e.value.replace(',', '.')) || 0;
		sum += e.parentNode.dataset.unit ? new Converter(n, e.parentNode.dataset.unit).value : n;
		num++;
	};
	return sum / (avg ? num : 1);
}

// Writes a number to all matching selectors. If the element has a data-unit
// type, converts the value to data datatype. Highlights the elements written to.
// Numbers under 100 gets precision 3, otherwise only writes whole numbers.
function putn(selector, value) {
	let color = getComputedStyle(document.querySelector(":root")).getPropertyValue("--calc-color");
	document.querySelectorAll(selector).forEach((e) => {
		let v = e.parentNode.dataset.unit ? new Converter(value).to_unit(e.parentNode.dataset.unit) : value;
		e.value = (v >= 100) ? v.toFixed(0) : parseFloat(v.toPrecision(3));
		e.parentNode.style.backgroundColor = "transparent";
		e.parentNode.animate(
			[
				{ backgroundColor: color },
				{ backgroundColor: color, offset: 0.8 },
				{ backgroundColor: "transparent" }
			], {
				duration: 5000
			}
		);
		e.addEventListener("click", function () {
			e.parentNode.style.backgroundColor = "";
		}, {once: true});
	});
}

// Returns the unit of the first element matching the selector
function getu(selector) {
	return document.querySelector(selector).parentNode.dataset.unit;
}

// Gets heatcap from the table, and converts it to Wh/m3K
function getmedia(selector) {
	return fm.heatcap[document.querySelector(selector).value];
}

class Calc {
	// callback must be the same as the instance name
	// definition is a object with the following structure:
	// variable: { id: <- the css id of the input
	//             f: <- function that returns the value, can be omitted if not calculable.
	//                   function needs to take a object with the variables as keys and numbers as values
	//             r: <- function that runs onkeyup, doesn't expect a return so you need to call putn from the function
	//             d: <- list of dependencys that needs to be resolved before calculation.
	//             c: <- list of calculatons to do after
	constructor(callback, definitions) {
		this.definitions = definitions;
		this.callback = callback;
		// Add buttons
		for(const [key, def] of Object.entries(this.definitions)) {
			if ("f" in def) {
				let button = document.createElement("button");
				button.type = "button";
				button.setAttribute("onclick", `${this.callback}.calc("${def.id}");`);
				button.className = "calc-button"
				button.innerHTML = "=";
				document.querySelector(def.id).parentNode.appendChild(button);
			}
			if ("r" in def) {
				let elem = document.querySelector(def.id);
				elem.setAttribute((elem.nodeName == "SELECT") ? "onchange" : "onkeyup", `${this.callback}.run("${def.id}");`);
			}
		}
	}

	calc(id) {
		let v = {};
		let f;
		let d;
		let c;
		// Fetch inputs
		for (const [key, def] of Object.entries(this.definitions)) {
			if (def.id == id) {
				f = def.f;
				d = def.d;
				c = def.c;
			} else {
				v[key] = getn(def.id);
			}
		}
		// Calculate dependencies
		if (Array.isArray(d)) {
			d.forEach( (key) => {
				if (!v[key]) {
					let s = this.definitions[key].id;
					this.calc(s);
					v[key] = getn(s);
				}
			});
		}
		// Run calculation
		let res = f(v);
		putn(id, res);
		// Call post calculations
		if (res && Array.isArray(c)) {
			c.forEach( (key) => {
				this.calc(this.definitions[key].id);
			});
		}
	}

	run(id) {
		let v = {};
		let r;
		for (const [key, def] of Object.entries(this.definitions)) {
				if (def.id == id) {
						r = def.r;
				}
				v[key] = {};
				v[key].id = def.id;
				v[key].val = getn(def.id);
		}
		r(v);
	}
}

class Conv {
	constructor(callback, ids) {
		this.ids = ids;
		this.ids.forEach((id) => {
			document.querySelector(id).setAttribute("onkeyup", `${callback}.convert("${id}");`);
		});
	}

	convert(id) {
		let value = getn(id);
		this.ids.forEach((i) => {
			if (id != i) {putn(i, value);}
		});
	}
}

class Fetcher {
	constructor(callback, fromPage, fromInput) {
		this.callback = callback;
		this.fromPage = fromPage;
		this.fromInput = fromInput;
	}

	add(s) {
		let button = document.createElement("button");
		button.type = "button";
		button.setAttribute("onclick", `${this.callback}.fetch("${s.page}", "${s.target}");`);
		button.className = "calc-button";
		button.innerHTML = "<";
		document.querySelector(s.target).parentNode.appendChild(button);
	}

	fetch(returnPage, returnTarget) {
		openPage(this.fromPage);
		document.querySelector(`${this.fromInput} ~ button`).addEventListener("click", () => {
			putn(returnTarget, getn(this.fromInput));
			openPage(returnPage);
		}, {once: true});
	}
}

function init() {
	fm.wasLastHome = true;
	history.pushState({},"");
	window.addEventListener("popstate", (e) => {
		openPage("startMenu");
	});
	for (let e of document.getElementsByClassName("dataUnit")) {
		e.parentNode.dataset.unit = e.innerText;
	}
	dataListener(document);
	document.addEventListener("keydown", nextInput);
	fPSettings();
	loadStatic();
}

init();
